<?php
// File: /public/invest.php
require_once '../includes/db.php';
require_once '../includes/auth.php';
require_once '../includes/functions.php';

requireLogin();
$user = getCurrentUser($pdo);

// Fetch Plans
$stmt = $pdo->query("SELECT * FROM investment_plans");
$plans = $stmt->fetchAll();

// Handle Investment
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $plan_id = $_POST['plan_id'];
    $amount = $_POST['amount'];

    // Validate Plan
    $stmt = $pdo->prepare("SELECT * FROM investment_plans WHERE id = ?");
    $stmt->execute([$plan_id]);
    $plan = $stmt->fetch();

    if (!$plan) {
        setFlash('danger', 'Invalid plan selected.');
    } elseif ($amount < $plan['min_amount'] || $amount > $plan['max_amount']) {
        setFlash('danger', "Amount must be between " . formatMoney($plan['min_amount']) . " and " . formatMoney($plan['max_amount']));
    } elseif ($user['balance'] < $amount) {
        setFlash('danger', 'Insufficient balance. Please deposit funds.');
    } else {
        // Process Investment
        try {
            $pdo->beginTransaction();

            // 1. Deduct Balance
            $new_balance = $user['balance'] - $amount;
            $stmt = $pdo->prepare("UPDATE users SET balance = ? WHERE id = ?");
            $stmt->execute([$new_balance, $user['id']]);

            // 2. Create Investment Record
            $end_date = date('Y-m-d H:i:s', strtotime("+{$plan['duration_days']} days"));
            $stmt = $pdo->prepare("INSERT INTO investments (user_id, plan_id, amount, end_date, status) VALUES (?, ?, ?, ?, 'active')");
            $stmt->execute([$user['id'], $plan['id'], $amount, $end_date]);

            // 3. Log Transaction
            logTransaction($pdo, $user['id'], 'investment', $amount, "Invested in {$plan['name']}");

            $pdo->commit();
            setFlash('success', 'Investment successful! Your capital is now growing.');
            redirect('dashboard.php');

        } catch (Exception $e) {
            $pdo->rollBack();
            setFlash('danger', 'Investment failed: ' . $e->getMessage());
        }
    }
}

$page_title = "Invest Capital";
include '../includes/header.php';
?>

<div class="card mb-4">
    <h3>Current Balance: <span class="text-primary"><?php echo formatMoney($user['balance']); ?></span></h3>
    <p class="text-muted">Choose a plan to start earning ROI.</p>
</div>

<div class="stats-grid" style="grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));">
    <?php foreach ($plans as $plan): ?>
    <div class="stat-card" style="border-top: 4px solid var(--accent-blue);">
        <h3 class="mb-2"><?php echo htmlspecialchars($plan['name']); ?></h3>
        <div class="stat-value text-primary"><?php echo $plan['roi_percentage']; ?>% ROI</div>
        <p class="text-muted mb-3">After <?php echo $plan['duration_days']; ?> Days</p>
        
        <ul class="mb-4 text-sm text-secondary">
            <li><i class="fas fa-check text-success"></i> Min: <?php echo formatMoney($plan['min_amount']); ?></li>
            <li><i class="fas fa-check text-success"></i> Max: <?php echo formatMoney($plan['max_amount']); ?></li>
            <li><i class="fas fa-check text-success"></i> 24/7 Support</li>
        </ul>

        <form method="POST" action="">
            <input type="hidden" name="plan_id" value="<?php echo $plan['id']; ?>">
            <div class="form-group">
                <input type="number" name="amount" class="form-control" placeholder="Enter Amount" min="<?php echo $plan['min_amount']; ?>" max="<?php echo $plan['max_amount']; ?>" required>
            </div>
            <button type="submit" class="btn btn-primary btn-block">Invest Now</button>
        </form>
    </div>
    <?php endforeach; ?>
</div>

<?php include '../includes/footer.php'; ?>
